<?php

namespace app\controllers;

use Yii;
use yii\web\Controller;
use yii\filters\AccessControl;

use app\models\settings\PayoutSettings;
use app\models\settings\GameMessageSettings;
use app\models\settings\GameStateSettings;
use app\models\settings\ManualResultTimeSettings;

class SettingsController extends Controller
{
    /**
     * {@inheritdoc}
     */
    public function behaviors()
    {
        return [
            'access' => [
                'class' => AccessControl::class,
                'only' => ['payouts', 'game-message', 'game-state'],
                'rules' => [
                    [
                        'actions' => ['payouts', 'game-message', 'game-state'],
                        'allow' => true,
                        'roles' => ['@'],
                        'matchCallback' => function ($rule, $action) {
                            if($action->id == 'game-state') {
                                return  Yii::$app->user->can('status game settings');
                            } else if($action->id == 'payouts') {
                                return  Yii::$app->user->can('payout settings settings');
                            } else if($action->id == 'game-message') {
                                return  Yii::$app->user->can('game message settings');
                            }
                        }
                    ],
                ],
            ]
        ];
    }

    public function actionPayouts()
    {
        $model = new PayoutSettings();
        if( $this->request->isPost ) {
            if($model->load($this->request->post()) && $model->validate() ) {
                $model->saveSettings();
                Yii::$app->session->setFlash('success', 'Payouts settings saved successfully');
            }
        }

        return $this->render('payouts', [
            'model' => $model
        ]);
    }

    public function actionGameMessage()
    {
        $model = new GameMessageSettings();
        if( $this->request->isPost ) {
            if($model->load($this->request->post()) && $model->validate() ) {
                $model->saveSettings();
                Yii::$app->session->setFlash('success', 'Message settings saved successfully');
            }
        }

        return $this->render('game-message', [
            'model' => $model
        ]);
    }

    public function actionGameState()
    {
        $model = new GameStateSettings();
        $model->saveSettings();
        Yii::$app->session->setFlash('success', 'Game status - ' . $model->state);

        return $this->redirect(['/']);
    }

    public function actionManualResult()
    {
        $db = Yii::$app->db;

        if($this->request->isPost) {
            $result_1 = $this->request->post('result_1');
            $result_2 = $this->request->post('result_2');
            $result_3 = $this->request->post('result_3');

            if( $result_1 == '' ) {
                Yii::$app->session->setFlash('error', 'Result 1 can not be empty');
                return $this->redirect(['settings/manual-result']);
            }

            $result_1 = intval($result_1);
            if( $result_1 < 1 || $result_1 > 99  ) {
                Yii::$app->session->setFlash('error', 'Result 1 must be between 1 and 99');
                return $this->redirect(['settings/manual-result']);
            }

            if( $result_2 != '' ) {
                $result_2 = intval($result_2);
                if( $result_2 < 1 || $result_2 > 99  ) {
                    Yii::$app->session->setFlash('error', 'Result 2 must be between 1 and 99');
                    return $this->redirect(['settings/manual-result']);
                }
            }

            if( $result_3 != '' ) {
                $result_3 = intval($result_3);
                if( $result_3 < 1 || $result_3 > 99  ) {
                    Yii::$app->session->setFlash('error', 'Result 3 must be between 1 and 99');
                    return $this->redirect(['settings/manual-result']);
                }
            }


            $result_1 = intval($result_1);
            $result_2 = $result_2 != '' ? intval($result_2) : null;
            $result_3 = $result_3 != '' ? intval($result_3) : null;

            $result_current_session_id = Yii::$app->session->get('result_current_session_id');
            
            if(is_null($result_current_session_id)) {
                Yii::$app->session->setFlash('error', 'Sorry! Game is closed');
                return $this->redirect(['settings/manual-result']);
            }

            $db->createCommand()->update('session', [
                'current_session_result' => $result_1,
                'current_session_result_2' => $result_2,
                'current_session_result_3' => $result_3,
                'result_multiple' => $this->request->post('multiple')
            ], [
                'current_session_id' => $result_current_session_id
            ])->execute();


            Yii::$app->session->setFlash('success', 'Result published successfully');
            return $this->redirect(['dashboard/index']);
        }
        
        date_default_timezone_set('Asia/Kolkata');
        
        $now = new \DateTime();
        $start = new \DateTime(date('Y-m-d') . '09:00:01');
        $end = new \DateTime(date('Y-m-d') . '22:00:01');
            
        if(($now > $start) && ($now < $end)) {
            $sessionInfo = $db->createCommand("SELECT * FROM session")->queryOne();

            $currentSessionId = $sessionInfo['current_session_id'];
    
            $dateTime = new \DateTime();
            $dateTime->setTimestamp( intval($currentSessionId) );
    
            // add the current session id to user current session
            Yii::$app->session->set('result_current_session_id', $currentSessionId);
            
            $result_time = $dateTime->format('h:i A');
        } else {
            Yii::$app->session->remove('result_current_session_id');
            $result_time = '';
        }
       
        return $this->render('manual-result', [
            'result_time' => $result_time
        ]);
    }
    
    public function actionSyncResult()
    {
        $sync_end_point = $_ENV['API_URL'] . '/game/hydrate-game-result';
        $curl_handle=curl_init();
        curl_setopt($curl_handle,CURLOPT_URL, $sync_end_point);
        curl_setopt($curl_handle,CURLOPT_RETURNTRANSFER,1);
        $data = curl_exec($curl_handle);
        curl_close($curl_handle);
        if (empty($data)){
            Yii::$app->session->setFlash('error', 'Server Error');
        }
        else{
            $data = json_decode($data, true);
            Yii::$app->session->setFlash('success', $data['message']);
        }

        return $this->redirect(['/']);
    }

    public function actionManualResultTime()
    {
        $model = new ManualResultTimeSettings();

        if($this->request->isPost) {
            if($model->load($this->request->post()) && $model->validate() ) {
                $model->saveSettings();
                Yii::$app->session->setFlash('success', 'Message settings saved successfully');
            } else {
                Yii::$app->session->setFlash('error', $model->getFirstError('time_limit'));
            }
        }

        return $this->render('manual-result-time', [
            'model' => $model
        ]);
    }
}