<?php

namespace app\models;

use Yii;
use yii\db\ActiveRecord;
use yii\web\IdentityInterface;
use yii\base\NotSupportedException;
use yii\behaviors\BlameableBehavior;


use app\models\Role;

class User extends ActiveRecord implements IdentityInterface
{
    const STATUS_ACTIVE = '1';
    const STATUS_INACTIVE = '0';

    /**
     * {@inheritdoc}
     */
    public static function tableName()
    {
        return 'users';
    }

    public function init()
    {
        parent::init();
        $this->on(self::EVENT_BEFORE_INSERT, function ($event) {
            $model = $event->sender;
            if (!is_null($this->password)) {
                $model->generatePasswordHash($this->password);
            }
            $model->auth_key = uniqid();
        });
    }

    public function behaviors()
    {
        return [
            BlameableBehavior::class,
        ];
    }

    /**
     * {@inheritdoc}
     */
    public static function findIdentity($id)
    {
        return static::findOne($id);
    }

    /**
     * {@inheritdoc}
     */
    public static function findIdentityByAccessToken($token, $type = null)
    {
        throw new NotSupportedException('"findIdentityByAccessToken" is not implemented.');
    }

    /**
     * Finds user by username
     *
     * @param string $username
     * @return static|null
     */
    public static function findByUsername($username)
    {
        return self::find()->where(['username' => $username, 'is_active' => self::STATUS_ACTIVE])->one();
    }

    /**
     * {@inheritdoc}
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * {@inheritdoc}
     */
    public function getAuthKey()
    {
        return $this->auth_key;
    }

    /**
     * {@inheritdoc}
     */
    public function validateAuthKey($authKey)
    {
        return $this->auth_key === $authKey;
    }

    /**
     * Validates password
     *
     * @param string $password password to validate
     * @return bool if password provided is valid for current user
     */
    public function validatePassword($password)
    {
        return $this->password === md5($password);
    }

    /**
     * Generates password hash from password and sets it to the model
     *
     * @param string $password
     */
    public function generatePasswordHash($password)
    {
        $this->password = md5($password);
    }

    public function setRole($role)
    {
        $role = Role::findOne(['name' => $role]);
        if ($role) {
            $this->role_id = $role->id;
        }
    }

    public static function getAllTerminalUsers()
    {
        $terminalRole = Role::findOne(['name' => Role::ROLE_TERMINAL]);

        //$stockRole = Role::findOne(['id' => Yii::$app->user->identity->role_id]);
        if (Yii::$app->session->get('role') == Role::ROLE_STOCK_LIST) {
            $userId =  Yii::$app->user->id;
            $terminal_id_array = Yii::$app->db->createCommand("SELECT terminal_id FROM stocklist_terminal WHERE stocklist_id=$userId")->queryAll();
            return self::findAll(['role_id' => $terminalRole->id, 'id' => array_column($terminal_id_array, 'terminal_id')]);
        } else {
            return self::findAll(['role_id' => $terminalRole->id]);
        }
    }

    public static function getAllStockListUsers()
    {
        $stockListRole = Role::findOne(['name' => Role::ROLE_STOCK_LIST]);
        return self::findAll(['role_id' => $stockListRole->id]);
    }

    public function updateWalletAmountForTerminal($amount)
    {
        $terminal_id = $this->id;

        // now find terminal balance
        $sql = "SELECT users.wallet_amount, users.id AS stocklist_id
        FROM users WHERE id = (
            SELECT stocklist_id FROM stocklist_terminal WHERE terminal_id = :terminal_id
        )";

        $db = Yii::$app->db;

        $stocklistWalletInfo = $db->createCommand($sql, [':terminal_id' => $terminal_id])->queryOne();

        if($stocklistWalletInfo == false) {
            throw new \Exception("No stocklist is assigned for this terminal");
        }

        if($stocklistWalletInfo['wallet_amount'] < $amount) {
            throw new \Exception("Insufficient balance");
        }
      
        $transaction = $db->beginTransaction();
        try {
            // update stocklist balance
            $sql = "UPDATE users SET wallet_amount = wallet_amount + :amount WHERE id = :terminal_id";
            $db->createCommand($sql, [':amount' => floatval($amount), ':terminal_id' => $terminal_id])->execute();

            $db->createCommand()->insert('wallet_txn', [
                'user_id' => $terminal_id,
                'txn_id' => 'TXN'.strtoupper(uniqid()),
                'amount' => $amount,
                'purpose' => 'added terminal balance from stocklist '. $stocklistWalletInfo['stocklist_id'],
                'type' => 'cr',
                'created_at' => date('Y-m-d H:i:s')
            ])->execute();
    
            // update stocklist balance
            $sql = "UPDATE users SET wallet_amount = wallet_amount - :amount WHERE id = :stocklist_id";
            $db->createCommand($sql, [':amount' => floatval($amount), ':stocklist_id' => $stocklistWalletInfo['stocklist_id']])->execute();
    
            $db->createCommand()->insert('wallet_txn', [
                'user_id' => $stocklistWalletInfo['stocklist_id'],
                'txn_id' => 'TXN'.strtoupper(uniqid()),
                'amount' => $amount,
                'purpose' => 'duducted balance and added to terminal ' . $terminal_id,
                'type' => 'dr',
                'created_at' => date('Y-m-d H:i:s')
            ])->execute();

            $transaction->commit();
            return true;
        } catch (\Exception $e) {
            $transaction->rollBack();
            throw $e;
        } catch (\Throwable $e) {
            $transaction->rollBack();
            throw $e;
        }
    }

    public function updateWalletAmountForStocklist($amount)
    {
        // add terminal balance
        $prevAmt = $this->wallet_amount;
        $newAmount = floatval($amount) + floatval($prevAmt);
        $this->wallet_amount = $newAmount;

        Yii::$app->db->createCommand()->insert('wallet_txn', [
            'user_id' => $this->id,
            'txn_id' => 'TXN'.strtoupper(uniqid()),
            'amount' => $amount,
            'purpose' => 'added wallet balance',
            'type' => 'cr',
            'created_at' => date('Y-m-d H:i:s')
        ])->execute();

        return $this->save(false);
    }
}
